using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;

namespace BreadAndCheese {

#region Game Sprites

    public class BaseSprite {
        protected Texture2D spriteTexture;
        protected Rectangle spriteRectangle;

        /// <summary>
        /// aduje tekstur uywan w ramach danego sprajtu.
        /// </summary>
        /// <param name="inSpriteTexture">Uywana tekstura.</param>
        public void LoadTexture(Texture2D inSpriteTexture) {
            spriteTexture = inSpriteTexture;
        }

        /// <summary>
        /// aduje prostokt wyznaczajcy wymiary na potrzeby 
        /// operacji rysowania tego sprajtu ta.
        /// </summary>
        /// <param name="inSpriteRectangle">Prostokt, ktry ma zosta uyty.</param>
        public void SetRectangle(Rectangle inSpriteRectangle) {
            spriteRectangle = inSpriteRectangle;
        }

        /// <summary>
        /// Metoda wywoywana w momencie uruchamiania gry
        /// </summary>
        public virtual void StartGame() {
        }

        /// <summary>
        /// Metoda wywoywana w momencie zatrzymywania gry
        /// </summary>
        public virtual void EndGame() {
        }

        /// <summary>
        /// Rysuje tekstur na podstawie prostokta okrelajcego jej wymiary.
        /// </summary>
        /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty
        /// przez operacj rysowania.</param>
        public virtual void Draw(SpriteBatch spriteBatch) {
            spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
        }

        /// <summary>
        /// Aktualizuje zachowanie dla tytuu. Sprajt bazowy nie zawiera adnego
        /// zachowania aktualizacji, ale pozostae sprajty mog zawiera takie zachowania.
        /// </summary>
        /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
        public virtual void Update(BreadAndCheeseGame game) {
        }

        public BaseSprite(Texture2D inSpriteTexture, Rectangle inRectangle) {
            LoadTexture(inSpriteTexture);
            SetRectangle(inRectangle);
        }

    }

    public class TitleSprite : BaseSprite {
        /// <summary>
        /// Aktualizuje zachowanie dla tytuu. Metoda testuje stan przycisku A i uruchamia gr,
        /// jeli ten przycisk jest wcinity.
        /// </summary>
        /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
        public override void Update(BreadAndCheeseGame game) {
            if (game.GamePad1.Buttons.A == ButtonState.Pressed) {
                game.StartGame();
            }
        }

        public TitleSprite(Texture2D inSpriteTexture, Rectangle inRectangle)
        : base(inSpriteTexture, inRectangle) {
        }
    }

    public class MovingSprite : BaseSprite {
        protected float x;
        protected float y;
        protected float xSpeed;
        protected float ySpeed;

        protected float initialX;
        protected float initialY;

        protected float minDisplayX;
        protected float maxDisplayX;

        protected float minDisplayY;
        protected float maxDisplayY;

        public float XPos
        {
            get
            {
                return x;
            }
        }

        public float YPos
        {
            get
            {
                return y;
            }
        }

        /// <summary>
        /// Sprawdza ewentualne kolizje pomidzy danym sprajtem a pozostaymi obiektami.
        /// </summary>
        /// <param name="target">Prostokt reprezentujcy pooenie innego obiektu.</param>
        /// <returns>true, jeli prostokt target koliduje z danym obiektem</returns>
        public virtual bool CheckCollision(Rectangle target) {
            return spriteRectangle.Intersects(target);
        }

        /// <summary>
        /// Tworzy poruszajcy si sprajt
        /// </summary>
        /// <param name="inSpriteTexture">Tekstura uywana dla tego sprajtu</param>
        /// <param name="widthFactor">Rozmiar obiektu w grze wyraony jako uamek szerokoci
        /// ekranu.</param>
        /// <param name="ticksToCrossScreen">Szybko poruszania si obiektu wyraona w liczbie taktw (1/60 sekundy) 
        /// potrzebnych do pokonania caego ekranu.</param>
        /// <param name="inMinDisplayX">minimalna warto X</param>
        /// <param name="inMaxDisplayX">maksymalna warto X</param>
        /// <param name="inMinDisplayY">minimalna warto Y</param>
        /// <param name="inMaxDisplayY">maksymalna warto Y</param>
        /// <param name="initialX">pocztkowe pooenie sprajtu w poziomie (X)</param>
        /// <param name="initialY">pocztkowe pooenie sprajtu w pionie (Y)</param>
        public MovingSprite(
                Texture2D inSpriteTexture,
                float widthFactor,
                float ticksToCrossScreen,
                float inMinDisplayX,
                float inMaxDisplayX,
                float inMinDisplayY,
                float inMaxDisplayY,
                float inInitialX,
                float inInitialY)
        : base(inSpriteTexture, Rectangle.Empty) {
            minDisplayX = inMinDisplayX;
            minDisplayY = inMinDisplayY;
            maxDisplayX = inMaxDisplayX;
            maxDisplayY = inMaxDisplayY;
            initialX = inInitialX;
            initialY = inInitialY;

            float displayWidth = maxDisplayX - minDisplayX;

            spriteRectangle.Width = (int) ((displayWidth * widthFactor) + 0.5f);
            float aspectRatio =
                    (float) spriteTexture.Width / spriteTexture.Height;
            spriteRectangle.Height =
                    (int) ((spriteRectangle.Width / aspectRatio) + 0.5f);
            x = initialX;
            y = initialY;
            xSpeed = displayWidth / ticksToCrossScreen;
            ySpeed = xSpeed;
        }

        public override void StartGame() {
            x = initialX;
            y = initialY;
            spriteRectangle.X = (int) x;
            spriteRectangle.Y = (int) y;
            base.StartGame();
        }
    }

    public class BatSprite : MovingSprite {
        /// <summary>
        /// Aktualizuje pooenie paki zalenie od stanu pada.
        /// </summary>
        public override void Update(BreadAndCheeseGame game) {
            x = x + (xSpeed * game.GamePad1.ThumbSticks.Left.X);
            y = y - (ySpeed * game.GamePad1.ThumbSticks.Left.Y);
            spriteRectangle.X = (int) x;
            spriteRectangle.Y = (int) y;
        }

        public BatSprite(Texture2D inSpriteTexture,
                float widthFactor, float ticksToCrossScreen,
                float inMinDisplayX, float inMaxDisplayX,
                float inMinDisplayY, float inMaxDisplayY,
                float inInitialX, float inInitialY)
        : base(inSpriteTexture, widthFactor, ticksToCrossScreen,
        inMinDisplayX, inMaxDisplayX,
        inMinDisplayY, inMaxDisplayY,
        inInitialX, inInitialY) {
        }
    }

    public class BallSprite : MovingSprite {
        /// <summary>
        /// Aktualizuje pooenie piki. Obsuguje kolizje z pak
        /// i celami. 
        /// </summary>
        /// <param name="game">Gra, w ktrej wystpuje ta pika</param>
        public override void Update(BreadAndCheeseGame game) {
            x = x + xSpeed;
            y = y + ySpeed;

            spriteRectangle.X = (int) (x + 0.5f);
            spriteRectangle.Y = (int) (y + 0.5f);

            if (game.BreadBat.CheckCollision(spriteRectangle)) {
                // Paka trafia w pik.
                ySpeed = ySpeed * -1;
            }

            if (x + spriteRectangle.Width >= maxDisplayX) {
                // Pika zostaa trafiona z prawej strony.
                xSpeed = Math.Abs(xSpeed) * -1;
            }

            if (x <= minDisplayX) {
                // Pika zostaa trafiona z lewej strony.
                xSpeed = Math.Abs(xSpeed);
            }

            if (y + spriteRectangle.Height >= maxDisplayY) {
                // Pika zderzya si z doln krawdzi. Powoduje utrat ycia.
                ySpeed = Math.Abs(ySpeed) * -1;
                game.LoseLife();
            }

            if (y <= minDisplayY) {
                // Pika uderzya w grn cz ekranu.
                ySpeed = Math.Abs(ySpeed);
            }

            if (game.TomatoTargets.CheckCollision(spriteRectangle)) {
                game.UpdateScore(20);
                ySpeed = ySpeed * -1;
            }
        }

        public BallSprite(Texture2D inSpriteTexture,
                float widthFactor, float ticksToCrossScreen,
                float inMinDisplayX, float inMaxDisplayX,
                float inMinDisplayY, float inMaxDisplayY,
                float inInitialX, float inInitialY)
        : base(inSpriteTexture, widthFactor, ticksToCrossScreen,
        inMinDisplayX, inMaxDisplayX,
        inMinDisplayY, inMaxDisplayY,
        inInitialX, inInitialY) {
        }
    }

    public class TargetRowSprite : BaseSprite {
        private int numberOfTargets;
        private float targetWidth;
        private float targetHeight;
        private float targetStepFactor;
        private float targetHeightLimit;
        private Rectangle[] targets;
        private bool[] targetVisibility;

        private float minDisplayX;
        private float maxDisplayX;
        private float minDisplayY;
        private float maxDisplayY;
        private float displayHeight;
        private float displayWidth;

        /// <summary>
        /// Rozpoczyna gr. Konfiguruje cele i przygotowuje je do uycia.
        /// </summary>
        /// <param name="inNumberOfTargets">Liczba celw na ekranie.</param>
        /// <param name="inTargetStepFactor">Cz ekranu, o ktr cele zostan przesunite w d
        /// po zniszczeniu caego rzdu.</param>
        /// <param name="inMinDisplayX"></param>
        /// <param name="inMaxDisplayX"></param>
        /// <param name="inMinDisplayY"></param>
        /// <param name="inMaxDisplayY"></param>
        public TargetRowSprite(
                Texture2D inSpriteTexture,
                int inNumberOfTargets,
                float inTargetStepFactor,
                float inMinDisplayX,
                float inMaxDisplayX,
                float inMinDisplayY,
                float inMaxDisplayY)
        : base(inSpriteTexture, Rectangle.Empty) {
            numberOfTargets = inNumberOfTargets;
            targetStepFactor = inTargetStepFactor;

            minDisplayX = inMinDisplayX;
            minDisplayY = inMinDisplayY;
            maxDisplayX = inMaxDisplayX;
            maxDisplayY = inMaxDisplayY;

            displayWidth = maxDisplayX - minDisplayX;
            displayHeight = maxDisplayY - minDisplayY;

            targetWidth = displayWidth / numberOfTargets;

            float aspectRatio = (float) spriteTexture.Width / spriteTexture.Height;

            targetHeight = targetWidth / aspectRatio;

            targetHeightLimit = minDisplayY + ((maxDisplayY - minDisplayY) / 2);

            targets = new Rectangle[numberOfTargets];
            targetVisibility = new bool[numberOfTargets];

            for (int i = 0; i < numberOfTargets; i++) {
                targets[i].Width = (int) targetWidth;
                targets[i].Height = (int) targetHeight;

                targets[i].Y = (int) targetHeight;
                targets[i].X = (int) (minDisplayX + (i * targetWidth) + 0.5f);

                targetVisibility[i] = true;
            }
        }

        public override void StartGame() {
            targetHeight = minDisplayY;

            // Przywraca ustawienia wszystkich celw.
            for (int i = 0; i < numberOfTargets; i++) {
                targets[i].Y = (int) targetHeight;
                targetVisibility[i] = true;
            }

            base.StartGame();
        }

        /// <summary>
        /// Aktualizuje stan celw na ekranie. Jeli wszystkie cele zostay zniszczone
        /// pooenie rzdu celw jest zmieniane, a same cele s ponownie wywietlane na ekranie.
        /// </summary>
        /// <param name="game">Gra zawierajca rzd celw. Ten parametr nie jest obecnie uywany
        /// ale moe zosta uyty w przyszoci, aby umoliwi celom przyspieszanie ruchu sera
        /// po kadym zniszczeniu caego rzdu celw.</param>
        public override void Update(BreadAndCheeseGame game) {
            for (int i = 0; i < numberOfTargets; i++) {
                if (targetVisibility[i]) {
                    // Zwraca sterowanie w razie znalezienia widocznego pomidora.
                    return;
                }
            }

            // Jeli sterowanie dotaro w to miejsce, nie ma widocznych pomidorw.

            // Pooenie rysowanego rzdu celw jest obniane.
            targetHeight = targetHeight + (displayHeight * targetStepFactor);

            // Sprawdza, czy osignito doln granic rysowania celw.
            if (targetHeight > targetHeightLimit) {
                targetHeight = minDisplayY;
            }

            // Przywraca ustawienia wszystkich celw.
            for (int i = 0; i < numberOfTargets; i++) {
                targets[i].Y = (int) targetHeight;
                targetVisibility[i] = true;
            }
        }

        /// <summary>
        /// Rysuje wszystkie widoczne cele.
        /// </summary>
        /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty podczas rysowania.</param>
        public override void Draw(SpriteBatch spriteBatch) {
            for (int i = 0; i < numberOfTargets; i++) {
                if (targetVisibility[i]) {
                    spriteBatch.Draw(spriteTexture, targets[i], Color.White);
                }
            }
        }

        /// <summary>
        /// Sprawdza, czy jaki obiekt nie koliduje z celem. Jeli tak,
        /// cel zostaje uznany za zniszczony.
        /// </summary>
        /// <param name="target">Pooenie celu.</param>
        /// <returns>true, jeli miaa miejsce kolizja i jeli cel zosta usunity.</returns>
        public bool CheckCollision(Rectangle target) {
            for (int i = 0; i < numberOfTargets; i++) {
                if (targetVisibility[i]) {
                    // Wykryto kolizj.
                    if (targets[i].Intersects(target)) {
                        // Cel jest niszczony.
                        targetVisibility[i] = false;
                        // Zwraca warto reprezentujc wystpienie kolizji.
                        return true;
                    }
                }
            }
            return false;
        }
    }

    public class DeadlySprite : MovingSprite {
        private bool isDeadly;


        public override void StartGame() {
            // Na pocztku gry papryki nie stanowi zagroenia.
            isDeadly = false;

            // Papryki pozostaj niegrone do momentu osignicia 200 punktw.
            deadlyTriggerScore = 200;

            base.StartGame();
        }

        public DeadlySprite(
                Texture2D inSpriteTexture,
                float widthFactor, float ticksToCrossScreen,
                float inMinDisplayX, float inMaxDisplayX, float inMinDisplayY,
                float inMaxDisplayY, float initialX, float initialY)
        : base(
        inSpriteTexture, widthFactor,
        ticksToCrossScreen, inMinDisplayX, inMaxDisplayX,
        inMinDisplayY, inMaxDisplayY, initialX, initialY) {
        }

        public override void Draw(SpriteBatch spriteBatch) {
            if (isDeadly) {
                spriteBatch.Draw(spriteTexture, spriteRectangle, Color.Red);
            } else {
                spriteBatch.Draw(spriteTexture, spriteRectangle, Color.Green);
            }
        }

        /// <summary>
        /// Liczba punktw, po ktrej jest wczany tryb miertelnego zagroenia.
        /// </summary>
        int deadlyScoreStep = 100;

        /// <summary>
        /// Nastpny poziom punktw, ktrego osignicie spowoduje przejcie w tryb miertelnego zagroenia.
        /// </summary>
        int deadlyTriggerScore = 200;

        /// <summary>
        /// Aktualizuje pooenie sprajtu. Obsuguje kolizje z pak
        /// i pik. 
        /// </summary>
        /// <param name="game">Gra, w ktrej wystpuje ta pika</param>
        public override void Update(BreadAndCheeseGame game) {
            // Sprajt zawsze odbija si od krawdzi ekranu i reaguje na kontakt
            // z pozostaymi obiektami, pod warunkiem e s widoczne.

            x = x + xSpeed;
            y = y + ySpeed;

            spriteRectangle.X = (int) (x + 0.5f);
            spriteRectangle.Y = (int) (y + 0.5f);

            if (x + spriteRectangle.Width >= maxDisplayX) {
                // Sprajt zderzy si z praw krawdzi.
                xSpeed = Math.Abs(xSpeed) * -1;
            }

            if (x <= minDisplayX) {
                // Sprajt zderzy si z lew krawdzi.
                xSpeed = Math.Abs(xSpeed);
            }

            if (y + spriteRectangle.Height >= maxDisplayY) {
                // Sprajt zderzy si z doln krawdzi.
                ySpeed = Math.Abs(ySpeed) * -1;
            }

            if (y <= minDisplayY) {
                // Papryka uderzya w grn krawd ekranu.
                ySpeed = Math.Abs(ySpeed);
            }

            if (game.GetScore() > deadlyTriggerScore) {
                // Wynik przekroczy prg.
                // Przechodzi w tryb miertelnego zagroenia i przesuwa prg.
                isDeadly = true;
                deadlyTriggerScore = deadlyTriggerScore + deadlyScoreStep;
            }

            if (isDeadly) {
                if (game.BreadBat.CheckCollision(spriteRectangle)) {
                    // Paka uderzya w ten sprajt.
                    isDeadly = false;
                    // utrata ycia
                    game.LoseLife();
                }

                if (game.CheeseBall.CheckCollision(spriteRectangle)) {
                    // pika zderzya si z tym sprajtem
                    isDeadly = false;
                    // aktualizuje liczb punktw
                    game.UpdateScore(50);
                }
            }
        }
    }

    public class KillerSprite : MovingSprite {
        private bool isDeadly;

        public override void StartGame() {
            isDeadly = false;
            killerTriggerScore = 50;
            base.StartGame();
        }

        /// <summary>
        /// Liczba punktw, po ktrej jest wczany tryb miertelnego zagroenia.
        /// </summary>
        int killerScoreStep = 400;

        /// <summary>
        /// Nastpny poziom punktw, ktrego osignicie spowoduje przejcie w tryb miertelnego zagroenia.
        /// </summary>
        int killerTriggerScore;

        public override void Update(BreadAndCheeseGame game) {

            if (game.GetScore() > killerTriggerScore) {
                // Wynik przekroczy prg.
                // Wcza zabjcze mandarynki i przesuwa prg.
                isDeadly = true;
                killerTriggerScore = killerTriggerScore + killerScoreStep;
            }

            if (isDeadly) {
                if (game.BreadBat.CheckCollision(spriteRectangle)) {
                    // paka zderzya z tym zabjczym sprajtem
                    isDeadly = false;
                    // odejmuje kilka punktw
                    game.UpdateScore(-10);
                }

                if (game.CheeseBall.CheckCollision(spriteRectangle)) {
                    // Pika trafia w zabjczy sprajt.
                    isDeadly = false;
                    // aktualizuje liczb punktw
                    game.UpdateScore(10);
                }

                if (game.BreadBat.XPos > x) {
                    x += xSpeed;
                } else {
                    x -= xSpeed;
                }
                if (game.BreadBat.YPos > y) {
                    y += ySpeed;
                } else {
                    y -= ySpeed;
                }
            }

            spriteRectangle.X = (int) (x + 0.5f);
            spriteRectangle.Y = (int) (y + 0.5f);

            base.Update(game);
        }

        public override void Draw(SpriteBatch spriteBatch) {
            if (isDeadly) {
                base.Draw(spriteBatch);
            }
        }


        static Random rand = new Random(1);

        public KillerSprite(
                Texture2D inSpriteTexture,
                float widthFactor, float ticksToCrossScreen,
                float inMinDisplayX, float inMaxDisplayX,
                float inMinDisplayY, float inMaxDisplayY,
                float inInitialX, float inInitialY)
        : base(inSpriteTexture, widthFactor, ticksToCrossScreen,
        inMinDisplayX, inMaxDisplayX,
        inMinDisplayY, inMaxDisplayY,
        inInitialX, inInitialY) {
        }
    }



#endregion

    /// <summary>
    /// To jest gwny typ Twojej gry.
    /// </summary>
    public class BreadAndCheeseGame : Microsoft.Xna.Framework.Game {
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;

        // wiat gry
        public GamePadState GamePad1;

        public BatSprite BreadBat;
        public BallSprite CheeseBall;
        public DeadlySprite DeadlyPepper;
        public TargetRowSprite TomatoTargets;
        public TitleSprite Title;
        public BaseSprite Background;

        public List<BaseSprite> GameSprites = new List<BaseSprite>();

        public KillerSprite KillerTangerine;

        /// <summary>
        /// Generator liczb losowych. Warto pocztkowa generatora jest staa,
        /// zatem losowe sekwencje zawsze bd takie same.
        /// </summary>
        private Random rand = new Random(1);

#region Wartoci reprezentujce wymiary ekranu

        int displayWidth;
        int displayHeight;
        float overScanPercentage = 10.0f;
        float minDisplayX;
        float maxDisplayX;
        float minDisplayY;
        float maxDisplayY;

        private void setScreenSizes() {
            displayWidth = graphics.GraphicsDevice.Viewport.Width;
            displayHeight = graphics.GraphicsDevice.Viewport.Height;
            float xOverscanMargin = getPercentage(overScanPercentage, displayWidth) / 2.0f;
            float yOverscanMargin = getPercentage(overScanPercentage, displayHeight) / 2.0f;

            minDisplayX = xOverscanMargin;
            minDisplayY = yOverscanMargin;

            maxDisplayX = displayWidth - xOverscanMargin;
            maxDisplayY = displayHeight - yOverscanMargin;
        }


#endregion

#region Metody pomocnicze

        /// <summary>
        /// Oblicza wartoci procentowe
        /// </summary>
        /// <param name="percentage">procent, ktry ma zosta wyznaczony</param>
        /// <param name="inputValue">warto do przeliczenia</param>
        /// <returns>warto wynikowa</returns>

        float getPercentage(float percentage, float inputValue) {
            return (inputValue * percentage) / 100;
        }

#endregion

#region Rysowanie tekstu

        SpriteFont font;

        private void loadFont() {
            font = Content.Load<SpriteFont > ("SpriteFont1");
        }

        /// <summary>
        /// Rysuje tekst na ekranie
        /// </summary>
        /// <param name="text">tekst do wywietlenia</param>
        /// <param name="textColor">kolor tekstu</param>
        /// <param name="x">lewa krawd tekstu</param>
        /// <param name="y">grna krawd tekstu</param>

        void drawText(string text, Color textColor, float x, float y) {
            int layer;
            Vector2 textVector = new Vector2(x, y);

            // Rysuje cie
            Color backColor = new Color(0, 0, 0, 10);
            for (layer = 0; layer < 10; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X++;
                textVector.Y++;
            }

            // Rysuje podstaw znakw
            backColor = new Color(190, 190, 190);
            for (layer = 0; layer < 5; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X++;
                textVector.Y++;
            }

            // Rysuje przedni cz znakw
            spriteBatch.DrawString(font, text, textVector, textColor);
        }

#endregion

#region Wywietlanie wyniku

        void drawScore() {
            drawText("Punkty: " + score + " ycia: " + lives, Color.Blue, minDisplayX, maxDisplayY - 50);
        }

        void drawHighScore() {
            drawText("Najlepszy wynik: " + highScore + " Nacinij A, aby rozpocz", Color.Blue, minDisplayX, minDisplayY);
        }

#endregion

#region Zarzdzanie gr i punktacj

        int score;
        int lives;
        int highScore;

        public enum GameState {
            titleScreen,
            playingGame
        }

        GameState state = GameState.titleScreen;

        public void StartGame() {
            score = 0;
            lives = 3;

            foreach(BaseSprite sprite in GameSprites) {
                sprite.StartGame();
            }

            state = GameState.playingGame;
        }

        void gameOver() {
            if (score > highScore) {
                highScore = score;
            }
            state = GameState.titleScreen;
        }

        public void LoseLife() {
            lives--;
            if (lives == 0) {
                gameOver();
            }
        }

        public void UpdateScore(int update) {
            score += update;
        }

        public int GetScore() {
            return score;
        }

        public int GetLives() {
            return lives;
        }

        public GameState GetState() {
            return state;
        }

#endregion


        public BreadAndCheeseGame() {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
        }

        /// <summary>
        /// Umoliwia ewentualn inicjalizacj przed uruchomieniem waciwej gry.
        /// W tym miejscu mona odnale wszystkie wymagane zasoby i zaadowa treci
        /// related content. Wywoanie metody base.Initialize spowoduje przeszukanie wszystkich komponentw
        /// i ich inicjalizacj.
        /// </summary>
        protected override void Initialize() {
            setScreenSizes();
            base.Initialize();
        }

        /// <summary>
        /// Metoda LoadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu adujcego tre.
        /// </summary>
        protected override void LoadContent() {
            // Tworzy nowy obiekt klasy SpriteBatch, ktrego mona uywa do rysowania tekstur.
            spriteBatch = new SpriteBatch(GraphicsDevice);

            Background = new BaseSprite(
                    Content.Load<Texture2D > ("Images/Background"),
                    new Rectangle((int) minDisplayX, (int) minDisplayY,
                    (int) (maxDisplayX - minDisplayX),
                    (int) (maxDisplayY - minDisplayY)
                    ));

            GameSprites.Add(Background);

            Title = new TitleSprite(
                    Content.Load<Texture2D > ("Images/Title"),
                    new Rectangle((int) minDisplayX, (int) minDisplayY,
                    (int) (maxDisplayX - minDisplayX),
                    (int) (maxDisplayY - minDisplayY)
                    ));

            CheeseBall = new BallSprite(
                    Content.Load<Texture2D > ("Images/Cheese"),
                    0.07f, // szeroko sera wynosi 0,07 szerokoci ekranu
                    200, // ser potrzebuje 200 taktw na pokonanie caego ekranu
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    displayWidth / 4, // jedna czwarta szerokoci ekranu od lewej krawdzi
                    displayHeight / 4); // jedna czwarta wysokoci ekranu od grnej krawdzi

            GameSprites.Add(CheeseBall);

            BreadBat = new BatSprite(
                    Content.Load<Texture2D > ("Images/Bread"),
                    0.166f, // szeroko chleba wynosi 0,166 szerokoci ekranu
                    150, // pokonanie ekranu zajmuje 150 taktw
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    displayWidth / 2, // na pocztku znajduje si na rodku ekranu (w poziomie)
                    displayHeight / 2); // na pocztku znajduje si na rodku ekranu (w pionie)

            GameSprites.Add(BreadBat);

            DeadlyPepper = new DeadlySprite(
                    Content.Load<Texture2D > ("Images/Pepper"),
                    0.1f,
                    500,
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    minDisplayX, minDisplayY);

            GameSprites.Add(DeadlyPepper);

            TomatoTargets = new TargetRowSprite(
                    Content.Load<Texture2D > ("Images/Tomato"),
                    20,
                    0.1f,
                    minDisplayX,
                    maxDisplayX,
                    minDisplayY,
                    maxDisplayY);

            GameSprites.Add(TomatoTargets);

            for (int i = 0; i < 100; i++) {
                KillerSprite Tangerine;

                float initialX = rand.Next((int) minDisplayX, (int) (maxDisplayX));
                float initialY = rand.Next((int) minDisplayY, (int) (maxDisplayY));
                int speed = 500 + rand.Next(500);

                Tangerine = new KillerSprite(
                        Content.Load<Texture2D > ("Images/Tangerine"),
                        0.03f,
                        speed,
                        minDisplayX,
                        maxDisplayX,
                        minDisplayY,
                        maxDisplayY,
                        initialX,
                        initialY);
                GameSprites.Add(Tangerine);
            }

            loadFont();
        }

        /// <summary>
        /// Metoda UnloadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu usuwajcego ca tre z pamici.
        /// </summary>
        protected override void UnloadContent() {
            // TODO: Naley usun ca tre, ktr nie zarzdza ContentManager.
        }

        /// <summary>
        /// Umoliwia grze wykonywanie logiki zwizanej z aktualizacj wiata gry,
        /// sprawdzaniem kolizji, pobieraniem danych wejciowych czy odtwarzaniem dwikw.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Update(GameTime gameTime) {
            GamePad1 = GamePad.GetState(PlayerIndex.One);

            if (GamePad1.Buttons.Back == ButtonState.Pressed)
                this.Exit();

            switch (state) {
                case GameState.titleScreen:
                    Title.Update(this);
                    break;
                case GameState.playingGame:
                    foreach(BaseSprite sprite in GameSprites)
                {
                    sprite.Update(this);
                }
                    break;
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// Ta metoda jest wywoywana w momencie, w ktrym gra musi narysowa swj wiat.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Draw(GameTime gameTime) {
            GraphicsDevice.Clear(Color.CornflowerBlue);

            spriteBatch.Begin();

            switch (state) {
                case GameState.titleScreen:
                    Title.Draw(spriteBatch);
                    drawHighScore();
                    break;
                case GameState.playingGame:
                    foreach(BaseSprite sprite in GameSprites)
                {
                    sprite.Draw(spriteBatch);
                }
                    drawScore();
                    break;
            }

            spriteBatch.End();

            base.Draw(gameTime);
        }
    }
}
